<?php

namespace App\Http\Controllers\SuperAdmin;

use App\Exports\SuperAdmin\TherapistCommunityPodcastsExport;
use App\Http\Controllers\Controller;
use App\Http\Requests\SuperAdmin\TherapistCommunityPodcasts\CreateRequest;
use App\Http\Requests\ImportRequest;
use App\Imports\SuperAdmin\TherapistCommunityPodcastsImport;
use App\Models\Podcast;
use App\Models\TherapistCommunity;
use App\Models\Tag;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Str;

class TherapistCommunityPodcastsController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

    /********* Initialize Permission based Middlewares  ***********/
    public function __construct()
    {
        $this->middleware('permission:therapist_community.add_podcast');
        $this->middleware('permission:therapist_community.add_podcast', ['only' => ['store']]);
        $this->middleware('permission:therapist_community.add_podcast', ['only' => ['update']]);
        $this->middleware('permission:therapist_community.add_podcast', ['only' => ['destroy']]);
        $this->middleware('permission:therapist_community.add_podcast', ['only' => ['export']]);
        $this->middleware('permission:therapist_community.add_podcast', ['only' => ['import']]);
    }
    /********* Getter For Pagination, Searching And Sorting  ***********/
    public function getter($req = null, $export = null, $therapist_community)
    {
        if ($req != null) {
            $therapist_community_podcasts =  $therapist_community->therapist_community_podcasts();
            if ($req->trash && $req->trash == 'with') {
                $therapist_community_podcasts =  $therapist_community_podcasts->withTrashed();
            }
            if ($req->trash && $req->trash == 'only') {
                $therapist_community_podcasts =  $therapist_community_podcasts->onlyTrashed();
            }
            if ($req->column && $req->column != null && $req->search != null) {
                $therapist_community_podcasts = $therapist_community_podcasts->whereLike($req->column, $req->search);
            } else if ($req->search && $req->search != null) {

                $therapist_community_podcasts = $therapist_community_podcasts->whereLike(['name', 'description'], $req->search);
            }
            if ($req->sort_field != null && $req->sort_type != null) {
                $therapist_community_podcasts = $therapist_community_podcasts->OrderBy($req->sort_field, $req->sort_type);
            } else {
                $therapist_community_podcasts = $therapist_community_podcasts->OrderBy('id', 'desc');
            }
            if ($export != null) { // for export do not paginate
                $therapist_community_podcasts = $therapist_community_podcasts->get();
                return $therapist_community_podcasts;
            }
            $therapist_community_podcasts = $therapist_community_podcasts->get();
            return $therapist_community_podcasts;
        }
        $therapist_community_podcasts = $therapist_community->therapist_community_podcasts()->withAll()->orderBy('id', 'desc')->get();
        return $therapist_community_podcasts;
    }


    /*********View All TherapistCommunityPodcasts  ***********/
    public function index(Request $request, TherapistCommunity $therapist_community)
    {
        $therapist_community_podcasts = $this->getter($request, null, $therapist_community);
        return view('super_admins.therapist_communitys.therapist_community_podcasts.index', compact('therapist_community_podcasts', 'therapist_community'));
    }

    /*********View Create Form of Podcast  ***********/
    public function create(TherapistCommunity $therapist_community)
    {
        $tags = Tag::active()->get();
        return view('super_admins.therapist_communitys.therapist_community_podcasts.create', compact('therapist_community', 'tags'));
    }

    /*********Store Podcast  ***********/
    public function store(CreateRequest $request, TherapistCommunity $therapist_community)
    {
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                $data['audio'] = uploadFile($request, 'file', 'therapist_community_podcasts');
            } else {
                $data['video'] = uploadFile($request, 'file', 'therapist_community_podcasts');
            }
            //$data['image'] = uploadCroppedFile($request,'image','therapist_community_podcasts');
            $therapist_community_podcast = $therapist_community->therapist_community_podcasts()->create($data);
            $therapist_community_podcast->slug = Str::slug($therapist_community_podcast->name . ' ' . $therapist_community_podcast->id, '-');
            $therapist_community_podcast->save();
            $therapist_community_podcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_podcasts.index', $therapist_community->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_podcasts.index', $therapist_community->id)->with('message', 'Podcast Created Successfully')->with('message_type', 'success');
    }

    /*********View Podcast  ***********/
    public function show(TherapistCommunity $therapist_community, Podcast $therapist_community_podcast)
    {
        if ($therapist_community->id != $therapist_community_podcast->therapist_community_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapist_communitys.therapist_community_podcasts.show', compact('therapist_community_podcast', 'therapist_community'));
    }

    /*********View Edit Form of Podcast  ***********/
    public function edit(TherapistCommunity $therapist_community, Podcast $therapist_community_podcast)
    {
        if ($therapist_community->id != $therapist_community_podcast->therapist_community_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        return view('super_admins.therapist_communitys.therapist_community_podcasts.edit', compact('therapist_community_podcast', 'therapist_community'));
    }

    /*********Update Podcast  ***********/
    public function update(CreateRequest $request, TherapistCommunity $therapist_community, Podcast $therapist_community_podcast)
    {
        if ($therapist_community->id != $therapist_community_podcast->therapist_community_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        $data = $request->all();
        try {
            DB::beginTransaction();
            if (!$request->is_active) {
                $data['is_active'] = 0;
            }
            if ($request->link_type == 'internal' && $request->file_type == 'audio') {
                if ($request->file) {
                    $data['audio'] = uploadFile($request, 'file', 'therapist_community_podcasts');
                } else {
                    $data['audio'] = $therapist_community_podcast->audio;
                }
            } else {
                if ($request->file) {
                    $data['video'] = uploadFile($request, 'file', 'therapist_community_podcasts');
                } else {
                    $data['video'] = $therapist_community_podcast->video;
                }
            }
            $therapist_community_podcast->update($data);
            $therapist_community_podcast = Podcast::find($therapist_community_podcast->id);
            $slug = Str::slug($therapist_community_podcast->name . ' ' . $therapist_community_podcast->id, '-');
            $therapist_community_podcast->update([
                'slug' => $slug
            ]);
            $therapist_community_podcast->tags()->sync($request->tag_ids);
            DB::commit();
        } catch (\Exception $e) {
            DB::rollback();
            return redirect()->route('super_admin.therapist_community_podcasts.index', $therapist_community->id)->with('message', 'Something Went Wrong')->with('message_type', 'error');
        }
        return redirect()->route('super_admin.therapist_community_podcasts.index', $therapist_community->id)->with('message', 'Podcast Updated Successfully')->with('message_type', 'success');
    }

    /********* Export  CSV And Excel  **********/
    public function export(Request $request)
    {
        $therapist_community_podcasts = $this->getter($request, "export");
        if (in_array($request->export, ['csv,xlsx'])) {
            $extension = $request->export;
        } else {
            $extension = 'xlsx';
        }
        $filename = "therapist_community_podcasts." . $extension;
        return Excel::download(new TherapistCommunityPodcastsExport($therapist_community_podcasts), $filename);
    }
    /********* Import CSV And Excel  **********/
    public function import(ImportRequest $request)
    {
        $file = $request->file('file');
        Excel::import(new TherapistCommunityPodcastsImport, $file);
        return redirect()->back()->with('message', 'Podcast Categories imported Successfully')->with('message_type', 'success');
    }


    /*********Soft DELETE Podcast ***********/
    public function destroy(TherapistCommunity $therapist_community, Podcast $therapist_community_podcast)
    {
        if ($therapist_community->id != $therapist_community_podcast->therapist_community_id) {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
        $therapist_community_podcast->delete();
        return redirect()->back()->with('message', 'Podcast Deleted Successfully')->with('message_type', 'success');
    }

    /*********Permanently DELETE Podcast ***********/
    public function destroyPermanently(Request $request, TherapistCommunity $therapist_community, $therapist_community_podcast)
    {
        $therapist_community_podcast = Podcast::withTrashed()->find($therapist_community_podcast);
        if ($therapist_community_podcast) {
            if ($therapist_community_podcast->trashed()) {
                if ($therapist_community_podcast->image && file_exists(public_path($therapist_community_podcast->image))) {
                    unlink(public_path($therapist_community_podcast->image));
                }
                $therapist_community_podcast->forceDelete();
                return redirect()->back()->with('message', 'Podcast Deleted Successfully')->with('message_type', 'success');
            } else {
                return redirect()->back()->with('message', 'Podcast is Not in Trash')->with('message_type', 'error');
            }
        } else {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
    }
    /********* Restore Podcast***********/
    public function restore(Request $request, TherapistCommunity $therapist_community, $therapist_community_podcast)
    {
        $therapist_community_podcast = Podcast::withTrashed()->find($therapist_community_podcast);
        if ($therapist_community_podcast->trashed()) {
            $therapist_community_podcast->restore();
            return redirect()->back()->with('message', 'Podcast Restored Successfully')->with('message_type', 'success');
        } else {
            return redirect()->back()->with('message', 'Podcast Not Found')->with('message_type', 'error');
        }
    }
}
